/*
  ==============================================================================

    SonicCrypt Seq
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"

NewProjectAudioProcessor::NewProjectAudioProcessor()
    : AudioProcessor(BusesProperties()
        .withInput("Input", juce::AudioChannelSet::stereo(), true)
        .withOutput("Output", juce::AudioChannelSet::stereo(), true)
    )
{
    auto folder = getPresetFolder();
    if (!folder.exists()) folder.createDirectory();
}

NewProjectAudioProcessor::~NewProjectAudioProcessor() {}

// --- MIDI EXPORT (GRID PERFECT) ---
juce::File NewProjectAudioProcessor::createMidiFileForExport(int numBars)
{
    juce::MidiMessageSequence track;
    const int ppq = 960;
    double totalLengthTicks = (4.0 * ppq) * numBars;

    std::array<LayerRuntime, 3> simRuntime;
    for (auto& rt : simRuntime) {
        rt.currentStep = 0;
        rt.currentOctave = 0;
        rt.samplesProcessed = 0;
        rt.isMovingForward = true;
    }

    // --- GLOBAL SOLO CHECK ---
    bool anySoloActive = false;
    for (int i = 0; i < 3; ++i) if (mySequencer.layers[i].settings.isSoloed) anySoloActive = true;

    for (int i = 0; i < 3; ++i) {
        auto& layer = mySequencer.layers[i];
        if (!layer.settings.isActive) continue;

        // Mute/Solo im Export
        if (anySoloActive && !layer.settings.isSoloed) continue;
        if (!anySoloActive && layer.settings.isMuted) continue;

        auto& rt = simRuntime[i];
        if (layer.settings.direction == 1) rt.currentStep = layer.settings.numSteps - 1;

        double ticksPerStep = 240.0; // Default 1/16
        bool allowSwing = true;

        switch (layer.settings.rateIndex) {
        case 0: ticksPerStep = 960.0; break;
        case 1: ticksPerStep = 480.0; break;
        case 2: ticksPerStep = 240.0; break;
        case 3: ticksPerStep = 120.0; break;
        case 4: ticksPerStep = 320.0; allowSwing = false; break;
        case 5: ticksPerStep = 160.0; allowSwing = false; break;
        }

        double currentTick = 0.0;
        int stepCounterForSwing = 0;
        int midiChannel = i + 1;

        while (currentTick < totalLengthTicks)
        {
            double stepDur = ticksPerStep;
            if (allowSwing && layer.settings.swing > 0.01f) {
                double swingPixels = ticksPerStep * layer.settings.swing;
                if (stepCounterForSwing % 2 == 0) stepDur += swingPixels;
                else stepDur -= swingPixels;
            }

            int stepIdx = rt.currentStep;
            auto& step = layer.steps[stepIdx];

            if (step.active) {
                int rootBase = 60 + layer.settings.rootNote;
                int octShift = rt.currentOctave * 12;
                int transpose = layer.settings.transpose;
                int interval = SequencerData::quantizeNote(step.noteOffset, layer.settings.scaleType);

                int note = juce::jlimit(0, 127, rootBase + interval + octShift + transpose);

                double noteOnPos = currentTick;
                double noteLen = stepDur * layer.settings.gateLength;

                if (noteOnPos + noteLen > totalLengthTicks) noteLen = totalLengthTicks - noteOnPos;

                if (noteLen > 1.0) {
                    auto onMsg = juce::MidiMessage::noteOn(midiChannel, note, step.velocity);
                    onMsg.setTimeStamp((int)(noteOnPos + 0.5));
                    track.addEvent(onMsg);

                    auto offMsg = juce::MidiMessage::noteOff(midiChannel, note);
                    offMsg.setTimeStamp((int)(noteOnPos + noteLen + 0.5));
                    track.addEvent(offMsg);
                }
            }

            // Logic to advance step
            int dir = layer.settings.direction;
            int nSteps = layer.settings.numSteps;
            int octRange = layer.settings.octaveRange;

            auto advOct = [&]() { rt.currentOctave++; if (rt.currentOctave >= octRange) rt.currentOctave = 0; };

            if (dir == 0) {
                rt.currentStep++;
                if (rt.currentStep >= nSteps) { rt.currentStep = 0; advOct(); }
            }
            else if (dir == 1) {
                rt.currentStep--;
                if (rt.currentStep < 0) { rt.currentStep = nSteps - 1; advOct(); }
            }
            else if (dir == 2) {
                if (rt.isMovingForward) {
                    rt.currentStep++;
                    if (rt.currentStep >= nSteps) {
                        rt.currentStep = nSteps - 2;
                        if (rt.currentStep < 0) rt.currentStep = 0;
                        rt.isMovingForward = false;
                    }
                }
                else {
                    rt.currentStep--;
                    if (rt.currentStep < 0) {
                        rt.currentStep = 1;
                        if (rt.currentStep >= nSteps) rt.currentStep = 0;
                        rt.isMovingForward = true;
                        advOct();
                    }
                }
            }
            else if (dir == 3) {
                juce::Random r;
                rt.currentStep = r.nextInt(nSteps);
                if (octRange > 1) rt.currentOctave = r.nextInt(octRange); else rt.currentOctave = 0;
            }

            currentTick += stepDur;
            stepCounterForSwing++;
        }
    }

    track.updateMatchedPairs();
    track.sort();

    juce::MidiFile midiFile;
    midiFile.setTicksPerQuarterNote(ppq);
    midiFile.addTrack(track);

    juce::File tempFile = juce::File::getSpecialLocation(juce::File::tempDirectory).getChildFile("SonicCrypt_Export.mid");
    if (tempFile.exists()) tempFile.deleteFile();

    juce::FileOutputStream stream(tempFile);
    if (stream.openedOk()) midiFile.writeTo(stream);

    return tempFile;
}

// --- STANDARD ---
juce::File NewProjectAudioProcessor::getPresetFolder() const {
    auto docDir = juce::File::getSpecialLocation(juce::File::userDocumentsDirectory);
    return docDir.getChildFile("SonicCrypt").getChildFile("SCSeq").getChildFile("Presets");
}
void NewProjectAudioProcessor::savePreset(const juce::String& name) {
    auto folder = getPresetFolder();
    if (!folder.exists()) folder.createDirectory();
    auto file = folder.getChildFile(name).withFileExtension("xml");
    auto xml = mySequencer.toXml();
    xml->writeTo(file);
}
void NewProjectAudioProcessor::loadPresetFile(const juce::File& file) {
    if (file.existsAsFile()) {
        auto xml = juce::XmlDocument::parse(file);
        if (xml) mySequencer.fromXml(*xml);
    }
}

// --- STATE INFORMATION (Updated for UI Size) ---
void NewProjectAudioProcessor::getStateInformation(juce::MemoryBlock& destData) {
    auto xml = mySequencer.toXml();

    // Save UI Size
    xml->setAttribute("uiWidth", lastUIWidth);
    xml->setAttribute("uiHeight", lastUIHeight);

    copyXmlToBinary(*xml, destData);
}

void NewProjectAudioProcessor::setStateInformation(const void* data, int sizeInBytes) {
    std::unique_ptr<juce::XmlElement> xml(getXmlFromBinary(data, sizeInBytes));
    if (xml.get() != nullptr) {
        // FIX: Nutze den aktuellen Wert (lastUIWidth) als Fallback, nicht den Startwert (1150).
        // Das verhindert, dass alte Projekte die Größe zurücksetzen.
        lastUIWidth = xml->getIntAttribute("uiWidth", lastUIWidth);
        lastUIHeight = xml->getIntAttribute("uiHeight", lastUIHeight);

        // Sicherheitscheck: Verhindere zu kleine Fenster (z.B. Fehler beim Laden)
        if (lastUIWidth < 100) lastUIWidth = 1150;
        if (lastUIHeight < 100) lastUIHeight = 650;

        mySequencer.fromXml(*xml);
    }
}

const juce::String NewProjectAudioProcessor::getName() const { return JucePlugin_Name; }
bool NewProjectAudioProcessor::acceptsMidi() const { return true; }
bool NewProjectAudioProcessor::producesMidi() const { return true; }
bool NewProjectAudioProcessor::isMidiEffect() const { return true; }
double NewProjectAudioProcessor::getTailLengthSeconds() const { return 0.0; }
int NewProjectAudioProcessor::getNumPrograms() { return 1; }
int NewProjectAudioProcessor::getCurrentProgram() { return 0; }
void NewProjectAudioProcessor::setCurrentProgram(int index) { juce::ignoreUnused(index); }
const juce::String NewProjectAudioProcessor::getProgramName(int index) { juce::ignoreUnused(index); return {}; }
void NewProjectAudioProcessor::changeProgramName(int index, const juce::String& newName) { juce::ignoreUnused(index, newName); }

void NewProjectAudioProcessor::prepareToPlay(double sampleRate, int samplesPerBlock) {
    juce::ignoreUnused(sampleRate, samplesPerBlock);
    for (auto& rt : layerRuntime) {
        rt.currentStep = 0; rt.currentOctave = 0;
        rt.samplesProcessed = 0; rt.isMovingForward = true;
    }
    activeInputNote = -1;
    activeNotes.clear();
}

void NewProjectAudioProcessor::releaseResources() {}

bool NewProjectAudioProcessor::isBusesLayoutSupported(const BusesLayout& layouts) const {
    if (layouts.getMainOutputChannelSet() != juce::AudioChannelSet::mono()
        && layouts.getMainOutputChannelSet() != juce::AudioChannelSet::stereo()) return false;
    return true;
}

void NewProjectAudioProcessor::processBlock(juce::AudioBuffer<float>& buffer, juce::MidiBuffer& midiMessages) {
    juce::ScopedNoDenormals noDenormals;
    buffer.clear();

    for (const auto metadata : midiMessages) {
        auto msg = metadata.getMessage();
        if (msg.isNoteOn()) {
            activeInputNote = msg.getNoteNumber();
            for (int i = 0; i < 3; ++i) {
                auto& rt = layerRuntime[i];
                auto& data = mySequencer.layers[i];
                rt.samplesProcessed = 0; rt.currentOctave = 0; rt.isMovingForward = true;
                if (data.settings.direction == 1) rt.currentStep = data.settings.numSteps - 1;
                else rt.currentStep = 0;
            }
        }
        else if (msg.isNoteOff() && msg.getNoteNumber() == activeInputNote) {
            activeInputNote = -1;
        }
    }
    midiMessages.clear();

    double samplesPerBeat = getSampleRate() * 60.0 / 120.0;
    if (auto* ph = getPlayHead()) {
        if (auto pos = ph->getPosition(); pos.hasValue())
            if (pos->getBpm().hasValue()) samplesPerBeat = getSampleRate() * 60.0 / *pos->getBpm();
    }

    int numSamples = buffer.getNumSamples();
    for (int s = 0; s < numSamples; ++s) {
        if (activeInputNote >= 0) {

            // --- GLOBAL SOLO CHECK ---
            bool anySoloActive = false;
            for (int i = 0; i < 3; ++i) if (mySequencer.layers[i].settings.isSoloed) anySoloActive = true;

            for (int i = 0; i < 3; ++i) {
                auto& layer = mySequencer.layers[i];
                if (!layer.settings.isActive) continue;

                // MUTE / SOLO Logic
                if (anySoloActive && !layer.settings.isSoloed) continue;
                if (!anySoloActive && layer.settings.isMuted) continue;

                auto& rt = layerRuntime[i];

                float multiplier = 0.25f;
                switch (layer.settings.rateIndex) {
                case 0: multiplier = 1.0f; break; case 1: multiplier = 0.5f; break;
                case 2: multiplier = 0.25f; break; case 3: multiplier = 0.125f; break;
                case 4: multiplier = 1.0f / 3.0f; break; case 5: multiplier = 1.0f / 6.0f; break;
                }

                int baseSamps = (int)(samplesPerBeat * multiplier);
                if (baseSamps < 1) baseSamps = 1;
                int stepDuration = baseSamps;
                int swingOffset = (int)(baseSamps * layer.settings.swing);
                if ((rt.currentStep % 2) == 0) stepDuration += swingOffset; else stepDuration -= swingOffset;
                if (stepDuration < 1) stepDuration = 1;

                if (rt.samplesProcessed == 0) {
                    if (rt.currentStep >= layer.settings.numSteps) rt.currentStep = 0;
                    if (rt.currentStep < 0) rt.currentStep = 0;
                    auto& step = layer.steps[rt.currentStep];
                    if (step.active) {
                        int transposeDelta = activeInputNote - 60;
                        int rootBase = 60 + layer.settings.rootNote;
                        int octShift = rt.currentOctave * 12;
                        int transpose = layer.settings.transpose;
                        int interval = SequencerData::quantizeNote(step.noteOffset, layer.settings.scaleType);

                        int finalNote = juce::jlimit(0, 127, rootBase + transposeDelta + interval + octShift + transpose);

                        midiMessages.addEvent(juce::MidiMessage::noteOn(1, finalNote, step.velocity), s);
                        int gateLen = (int)(stepDuration * layer.settings.gateLength);
                        activeNotes.push_back({ finalNote, gateLen });
                    }
                }
                rt.samplesProcessed++;
                if (rt.samplesProcessed >= stepDuration) {
                    rt.samplesProcessed = 0;
                    int dir = layer.settings.direction;
                    int octRange = layer.settings.octaveRange;
                    int numSteps = layer.settings.numSteps;
                    auto advOct = [&]() { rt.currentOctave++; if (rt.currentOctave >= octRange) rt.currentOctave = 0; };

                    if (dir == 0) { rt.currentStep++; if (rt.currentStep >= numSteps) { rt.currentStep = 0; advOct(); } }
                    else if (dir == 1) { rt.currentStep--; if (rt.currentStep < 0) { rt.currentStep = numSteps - 1; advOct(); } }
                    else if (dir == 2) {
                        if (rt.isMovingForward) { rt.currentStep++; if (rt.currentStep >= numSteps) { rt.currentStep = numSteps - 2; if (rt.currentStep < 0)rt.currentStep = 0; rt.isMovingForward = false; } }
                        else { rt.currentStep--; if (rt.currentStep < 0) { rt.currentStep = 1; if (rt.currentStep >= numSteps)rt.currentStep = 0; rt.isMovingForward = true; advOct(); } }
                    }
                    else if (dir == 3) {
                        juce::Random r; rt.currentStep = r.nextInt(numSteps);
                        if (octRange > 1) rt.currentOctave = r.nextInt(octRange); else rt.currentOctave = 0;
                    }
                }
            }
        }
        for (int k = activeNotes.size() - 1; k >= 0; --k) {
            activeNotes[k].samplesRemaining--;
            if (activeNotes[k].samplesRemaining <= 0) {
                midiMessages.addEvent(juce::MidiMessage::noteOff(1, activeNotes[k].noteNumber), s);
                activeNotes.erase(activeNotes.begin() + k);
            }
        }
    }
}

bool NewProjectAudioProcessor::hasEditor() const { return true; }
juce::AudioProcessorEditor* NewProjectAudioProcessor::createEditor() { return new NewProjectAudioProcessorEditor(*this); }
juce::AudioProcessor* JUCE_CALLTYPE createPluginFilter() { return new NewProjectAudioProcessor(); }